package u3d;

import java.io.*;
import java.nio.channels.*;
import java.nio.*;
import javax.vecmath.*;
import customnode.*;
import ij3d.*;
import javax.media.j3d.*;
import java.util.*;

/**
 * Some links:
 * http://www.ecma-international.org/publications/files/ECMA-ST/ECMA-363%204th%20Edition.pdf
 * http://www3.math.tu-berlin.de/jreality/download/jr/src-io/de/jreality/writer/u3d/
 * http://u3d.svn.sourceforge.net/viewvc/u3d/trunk/Source/Samples/Data/
 * http://www.ctan.org/tex-archive/macros/latex/contrib/movie15/
 * 
 */
public class U3DExporter {

	public static void main(String[] args) throws IOException {
		List<Point3f> l = customnode.MeshMaker.createSphere(10, 20, 15, 50f);
		CustomTriangleMesh ctm = new CustomTriangleMesh(l);
		ctm.setColor(new Color3f(0.5f, 0.5f, 0.1f));

		Image3DUniverse univ = new Image3DUniverse();
		univ.show();

		univ.addCustomMesh(ctm, "sphere");
		
		writeU3D(univ, "/tmp/sphere.u3d");
	}

	private final int uACContextBaseShadingID = 1;
	private final int uACStaticFull           = 0x00000400;
	private final int uACMaxRange             = uACStaticFull + 0x00003FFF;

	public void writeU3D(CustomTriangleMesh mesh, String name, String path) throws IOException {
		TriangleArray g = (TriangleArray)mesh.getGeometry();
		int N = g.getValidVertexCount();
		Point3f[] coords = new Point3f[N];
		Color3f[] colors = new Color3f[N];
		Vector3f[] normals = new Vector3f[N];
		for(int i = 0; i < N; i++) {
			coords[i] = new Point3f();
			colors[i] = new Color3f();
			normals[i] = new Vector3f();
		}
		g.getCoordinates(0, coords);
		g.getColors(0, colors);
		g.getNormals(0, normals);

		// create indices
		Map<Point3f, Integer>  vertexToIndex = new HashMap<Point3f, Integer>();
		Map<Color3f, Integer>  colorToIndex  = new HashMap<Color3f, Integer>();
		Map<Vector3f, Integer> normalToIndex = new HashMap<Vector3f, Integer>();

		int nFaces = N;
		int[] coordIndices  = new int[nFaces];
		int[] colorIndices  = new int[nFaces];
		int[] normalIndices = new int[nFaces];

		List<Point3f>  vList = new ArrayList<Point3f>();
		List<Color3f>  cList = new ArrayList<Color3f>();
		List<Vector3f> nList = new ArrayList<Vector3f>();

		for(int i = 0; i < N; i++) {
			Point3f v  = coords[i];
			Color3f c  = colors[i];
			Vector3f n = normals[i];

			if(!vertexToIndex.containsKey(v)) {
				Point3f newp = new Point3f(v);
				vertexToIndex.put(newp, vList.size());
				vList.add(newp);
			}
			coordIndices[i] = vertexToIndex.get(v);
			if(!colorToIndex.containsKey(c)) {
				Color3f newc = new Color3f(c);
				colorToIndex.put(newc, cList.size());
				cList.add(newc);
			}
			colorIndices[i] = colorToIndex.get(c);
			if(!normalToIndex.containsKey(n)) {
				Vector3f newn = new Vector3f(n);
				normalToIndex.put(newn, nList.size());
				nList.add(newn);
			}
			normalIndices[i] = normalToIndex.get(n);
		}

		coords = new Point3f[vList.size()];
		vList.toArray(coords);

		normals = new Vector3f[nList.size()];
		nList.toArray(normals);

		colors = new Color3f[cList.size()];
		cList.toArray(colors);


		Point3f min = new Point3f();
		Point3f max = new Point3f();
		Point3f center = new Point3f();
		mesh.calculateMinMaxCenterPoint(min, max, center);
		
		float dx = max.x - min.x;
		float dy = max.y - min.y;
		float dz = max.z - min.z;
		float maxd = Math.max(dx, Math.max(dy, dz));

		for(Point3f p : coords) {
			p.sub(center);
			p.scale(1 / maxd);
		}

		ByteArrayOutputStream bOut = new ByteArrayOutputStream();
		WritableByteChannel o = java.nio.channels.Channels.newChannel(bOut);
		int ds = 0;
		int cs = 0;
		DataBlock b;

		float[] matrix = new float[] {1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1};
		b = getNodeModifierChain("Box01", "Box01", "LightBoxModel", "Box01", "Box010", matrix);
		ds += writeDataBlock(b, o);

		b = getModelResourceModifierChain("LightBoxModel", coords, normals, colors, coordIndices, normalIndices, colorIndices, "LightBoxModel");
		ds += writeDataBlock(b, o);


		b = getMeshContinuationBlock(coords, normals, colors, coordIndices, normalIndices, colorIndices, "LightBoxModel");
		cs += writeDataBlock(b, o);



		// write header and data
		OutputStream out = new FileOutputStream(path);
		o = java.nio.channels.Channels.newChannel(out);
		writeDataBlock(getHeaderBlock(ds, cs), o);
		bOut.writeTo(out);
		out.close();
		System.out.println("done.");
	}

	private ByteBuffer buffer = ByteBuffer.allocate(1024 * 1024).order(ByteOrder.LITTLE_ENDIAN);


	private int writeDataBlock(DataBlock b, WritableByteChannel o) throws IOException {

	        int dataSize = (int)Math.ceil(b.getDataSize() / 4.0); // include padding
	        int metaDataSize = (int)Math.ceil(b.getMetaDataSize() / 4.0); // include padding

	        int blockLength = (int)(12 + 4 * (dataSize + metaDataSize));
	        if (buffer.capacity() < blockLength) {
	                buffer = ByteBuffer.allocate(blockLength);
	                buffer.order(ByteOrder.LITTLE_ENDIAN);
	        }
	        buffer.position(0);
	        buffer.limit(blockLength);

	        buffer.putInt((int)b.getBlockType());
	        buffer.putInt((int)b.getDataSize());
	        buffer.putInt((int)b.getMetaDataSize());

	        for (int i = 0; i < dataSize; i++)
	                buffer.putInt((int)b.getData()[i]);
	        for (int i = 0; i < metaDataSize; i++)
	                buffer.putInt((int)b.getMetaData()[i]);
	        buffer.rewind();
	        o.write(buffer);
	        return blockLength;
	}

	private DataBlock getHeaderBlock(int declSize, long contSize) {
		BitStreamWrite w = new BitStreamWrite();
		w.WriteI16((short)256);               // major version
	        w.WriteI16((short)0);                 // minor version
	        w.WriteU32(0);                        // profile identifier TODO
	        w.WriteU32(36 + declSize);            // declaration size
	        w.WriteU64(36 + declSize + contSize); // file size
	        w.WriteU32(106);                      // character encoding: 106 = UTF-8
	        DataBlock b = w.GetDataBlock();
	        b.setBlockType(0x00443355);           // file header block
	        return b;
	}

	private void WriteMatrix(BitStreamWrite w, float[] mat) {
		for(int i = 0; i < mat.length; i++)
			w.WriteF32(mat[i]);
	}

	private DataBlock getModelNodeBlock(String modelNodeName, String modelResourceName, float[] matrix) {
		BitStreamWrite w = new BitStreamWrite();
		long type = (0xffff << 16) | 0xff22;   // model node block
		w.WriteString(modelNodeName);          // model node name
		w.WriteU32(1);                         // parent node count
		w.WriteString("");                     // parent node name
		WriteMatrix(w, matrix);                // transformation
		w.WriteString(modelResourceName);      // model resource name
		w.WriteU32(1);                         // visibility 3 = front and back
		DataBlock b = w.GetDataBlock();
		b.setBlockType(type);
		return b;
	}

	private DataBlock getShadingModifierBlock(String shadingModName, String shaderName ) {
		BitStreamWrite w = new BitStreamWrite();
		long type = (0xffff << 16) | 0xff45;   // shading modifier block
		w.WriteString(shadingModName);         // shading modifier name
		w.WriteU32(1);                         // chain index
		w.WriteU32(15);                        // shading attributes
		w.WriteU32(1);                         // shading list count
		w.WriteU32(1);                         // shader count
		w.WriteString(shaderName);             // shader name
		DataBlock b = w.GetDataBlock();
		b.setBlockType(type);
		return b;
	}

	private DataBlock getNodeModifierChain(String modifierChainName, String modelNodeName, String modelResourceName, String shadingName, String shaderName, float[] matrix) {
		long type = (0xffff << 16) | 0xff14;  // modifier chain block
		BitStreamWrite w = new BitStreamWrite();
		w.WriteString(modifierChainName);     // modifier chain name
		w.WriteU32(0);                        // modifier chain type: 0 = node modifier chain
		w.WriteU32(0);                        // modifier chain attributes: 0 = neither bounding sphere nor bounding box info present
		w.AlignTo4Byte();
		w.WriteU32(2);                        // modifier count in this chain

		w.WriteDataBlock(getModelNodeBlock(modelNodeName, modelResourceName, matrix));
		w.WriteDataBlock(getShadingModifierBlock(shadingName, shaderName));

		DataBlock b = w.GetDataBlock();
		b.setBlockType(type);
		return b;
	}

	private DataBlock getMeshDeclarationBlock(Point3f[] coords, Vector3f[] normals, Color3f[] colors, int[] coordIndices, int[] normalIndices, int[] colorIndices, String meshname) {
		BitStreamWrite w = new BitStreamWrite();
		long type = (0xffff << 16) | 0xff31;  // mesh declaration block
		w.WriteString(meshname);              // mesh name
		w.WriteU32(0);                        // chain index
		// max mesh description
		w.WriteU32(0);                        // mesh attributes: 1 = no normals
		w.WriteU32(coordIndices.length / 3);  // face count
		w.WriteU32(coords.length);            // positions count
		w.WriteU32(normals.length);           // normal count
		w.WriteU32(colors.length);            // diffuse color count
		w.WriteU32(0);                        // specular color count
		w.WriteU32(0);                        // texture coord count
		w.WriteU32(1);                        // shading count
		// shading description
		w.WriteU32(1);                        // shading attributes: 0 = the shader list uses neither diffuse nor specular colors
		w.WriteU32(0);                        // texture layer count
		w.WriteU32(2);
		w.WriteU32(0);                        // original shader id
		// clod desc
		w.WriteU32(coords.length);            // minimum resolution
		w.WriteU32(coords.length);            // maximum resolution
		w.WriteU32(300);                      // position quality factor
		w.WriteU32(300);                      // normal quality factor
		w.WriteU32(300);                      // texture coord quality factor
		w.WriteF32(0.01f);                    // position inverse quant
		w.WriteF32(0.01f);                    // normal inverse quant
		w.WriteF32(0.01f);                    // texture coord inverse quant
		w.WriteF32(0.01f);                    // diffuse color inverse quant
		w.WriteF32(0.01f);                    // specular color inverse quant
		w.WriteF32(0.9f);                     // normal crease parameter
		w.WriteF32(0.5f);                     // normal update parameter
		w.WriteF32(0.985f);                   // normal tolerance parameter
		w.WriteU32(0);                        // bone count

		DataBlock b = w.GetDataBlock();
		b.setBlockType(type);
		return b;
	}

	private DataBlock getModelResourceModifierChain(String modifierChainName, Point3f[] coords, Vector3f[] normals, Color3f[] colors, int[] coordIndices, int[] normalIndices, int[] colorIndices, String meshname) {
		BitStreamWrite w = new BitStreamWrite();
		long type = (0xffff << 16) | 0xff14;  // modifier chain block
		w.WriteString(modifierChainName);     // modifier chain name
		w.WriteU32(1);                        // modifier chain type: 1 = model resource modifier chain
		w.WriteU32(0);                        // modifier chain attributes: 0 = neither bounding sphere nor bounding box info present
		// padding
		w.AlignTo4Byte();
		w.WriteU32(1);                        // modifier count in this chain

		w.WriteDataBlock(getMeshDeclarationBlock(coords, normals, colors, coordIndices, normalIndices, colorIndices, meshname));
		DataBlock b = w.GetDataBlock();
		b.setBlockType(type);
		return b;
	}

	private DataBlock getMeshContinuationBlock(Point3f[] coords, Vector3f[] normals, Color3f[] colors, int[] coordIndices, int[] normalIndices, int[] colorIndices, String meshname) {
		BitStreamWrite w = new BitStreamWrite();
		long type = (0xffff << 16) | 0xff3b;  // mesh continuation block
		w.WriteString(meshname);              // mesh name
		w.WriteU32(0);                        // chain index
		w.WriteU32(coordIndices.length / 3);  // base face count
		w.WriteU32(coords.length);            // base position count
		w.WriteU32(normals.length);           // base normal count
		w.WriteU32(colors.length);            // base diffuse color count
		w.WriteU32(0);                        // base specular color count
		w.WriteU32(0);                        // base texture coord count


		for(int i = 0; i < coords.length; i++) {
			w.WriteF32(coords[i].x);      // base position x
			w.WriteF32(coords[i].y);      // base position y
			w.WriteF32(coords[i].z);      // base position z
		}
		for(int i = 0; i < normals.length; i++) {
			w.WriteF32(normals[i].x);     // base normal x
			w.WriteF32(normals[i].y);     // base normal y
			w.WriteF32(normals[i].z);     // base normal z
		}
		for(int i = 0; i < colors.length; i++) {
			w.WriteF32(colors[i].x);     // base colors red
			w.WriteF32(colors[i].y);     // base colors green
			w.WriteF32(colors[i].z);     // base colors blue
			w.WriteF32(1);               // base colors alpha
		}

		for(int i = 0; i < coordIndices.length; i += 3) {
			w.WriteCompressedU32(uACContextBaseShadingID, 0);  // shading id
			for (int j = 0; j < 3; j++) {
		 		w.WriteCompressedU32(uACStaticFull + coords.length,  coordIndices [i + j]);
		 		w.WriteCompressedU32(uACStaticFull + normals.length, normalIndices[i + j]);
		 		w.WriteCompressedU32(uACStaticFull + colors.length,  colorIndices [i + j]);
			}
		}

		DataBlock b = w.GetDataBlock();
		b.setBlockType(type);
		return b;
	}
}
